#!/bin/bash
VER=1.1
#-[ Intro ]----------------------------------------------------#
#                                                              #
# Tur-AccountExpire. A system for setting a date for when the  #
# account will become active and/or inactive. Perfect if you   #
# add temporary users but always forget to del em manually.    #
#                                                              #
# It will NOT delete users. What it will do is add a custom    #
# flag. If you select this flag to be 6, you will have to      #
# purge them manually.                                         #
#                                                              #
# Writes to syslog (like normal gl siteop commands) and can    #
# send a "byefile" to the user if the account expires, so they #
# know why they are delled (only works for flag 6).            #
#                                                              #
#-[ Installation ]---------------------------------------------#
#                                                              #
# Copy tur-accountexpire.sh to /glftpd/bin. Make it executable #
# with 755 or similar.                                         #
#                                                              #
# Create the DB file and set chmod 777 on it (see settings).   #
#                                                              #
# Add to your glftpd.conf:                                     #
# site_cmd expire  EXEC  /bin/tur-accountexpire.sh             #
# custom-expire    1                                           #
#                                                              #
# Configure the settings below:                                #
#                                                              #
# GLROOT   = The path to your glftpd root. Usually /glftpd     #
#                                                              #
# USERSDIR = Path to your users dir, relative to GLROOT above. #
#                                                              #
# SYSLOG   = Path your your sysop.log file, relative to GLROOT #
#            above. Set to "" to disable logging or set a      #
#            different log file if you want.                   #
#                                                              #
# TMP      = A temporary dir, relative to GLROOT. Make sure    #
#            it exists and is chmod 777.                       #
#                                                              #
# BYEFILES = Path to the byefiles dir, relative to GLROOT.     #
#            This is only needed if USERFLAG is 6              #
#            "" = Disable it.                                  #
#                                                              #
# DB       = Path to the DB you want it to store info in.      #
#            Guess what? Its relative to GLROOT above.         #
#                                                              #
# DATEBIN  = Date binary to use. Most people can leave it      #
#            empty and it will just use 'date' in your path.   #
#            (make sure date is in /glftpd/bin)                #
#            FBSD users will need to download sh-utils and     #
#            specify the path to 'gdate' here as we need a GNU #
#            compliant date binary ( that can handle -d ).     #
#                                                              #
# USEFLAG  = This is the default flag to use for giving or     #
#            removing from users.                              #
#            You can set this to flag 6 and users will be      #
#            deleted. You may also, for instance, add flag     #
#            J or something. If you then add, to glftpd.conf:  #
#            shutdown !J *                                     #
#            It will say the site is closed when expired users #
#            try to log on. Feels a bit better perhaps.        #
#                                                              #
#            You may specify more then one flag here if needed #
#            ( USEFLAG="4I" for instance )"                    #
#                                                              #
#            This may be overridden with the f:<flag> option   #
#            when adding a user for expiration.                #
#                                                              #
# LINEUP   = When checking status and this is TRUE, it will    #
#            try to line up the fields. Guess this dosnt work  #
#            on all distros so made it an option.              #
#                                                              #
# PERSISTENT= If this is set to TRUE, it will, when running    #
#             the 'daily' procedure, see if any event was set  #
#             to occour in the past. If that is so, it will    #
#             check if the user still has (or hasnt) that flag #
#             and add or remove it now if not.                 #
#                                                              #
#             That means that if you set someone to get flag 4 #
#             on a certain date, it will check each day after  #
#             to make sure the user still has flag 4.          #
#             If someone removes it, he gets flag 4 again the  #
#             next time this script runs 'daily'.              #
#                                                              #
#             Important: This only works if EITHER start OR    #
#             end time is set. If both start and end is set on #
#             a user, it will set PERSISTENT="FALSE" for that  #
#             user.                                            #
#             Otherwise, wierd stuff will happen (add & remove #
#             each day etc).                                   #
#                                                              #
#             If you set this to FALSE, it will ONLY do this   #
#             check if its the exact date you set (default).   #
#                                                              #
#-[ Crontab ]--------------------------------------------------#
#                                                              #
# Is your crontab getting cluttered with tur-* scripts yet?    #
# Anyway, this is ment to be execute AFTER midnight each day.  #
# So add the following to run it every day, 1 minute midnight: #
# 1 0 * * *       /glftpd/bin/tur-accountexpire.sh daily       #
#                                                              #
#-[ Info ]-----------------------------------------------------#
#                                                              #
# Adding or removing users does not automatically give or      #
# remove the selected flag. You need to do that yourself.      #
# For instance, you use flag J and you want to enable someone  #
# in 5 days. You then need to manually add the flag first.     #
# After 5 days, it will be removed again.                      #
#                                                              #
# Example1 ( r = Remove flag when ):                           #
# site adduser test testing *@127.*                            #
# site change test flags +J                                    #
# site expire add test r:5                                     #
# User 'test' wont be able to login until after midnight, 5    #
# days from now.                                               #
#                                                              #
# Example2 ( a = Add flag when ):                              #
# site adduser test testing *@127.*                            #
# site expire add test a:5                                     #
# User 'test' will be active for 5 days from now and then he   #
# get flag J                                                   #
#                                                              #
# Example3 ( f = Custom flag to use ):                         #
# site adduser test testing *@127.*                            #
# site expire add test a:7 r:10 f:W                            #
# User 'test' will get flag W in 7 days and then loose it in   #
# 10 (from now).                                               #
#                                                              #
# Instead of # of days, you may set a static date, like        #
# r:2003-10-25 and the flag will be removed at that date.      #
#                                                              #
# You can not use todays date or any date in the past when     #
# adding people to the db with static date. For testing        #
# purposes, I allowed "0" to be used for number of days though.#
# (a:0). After that, you may run 'tur-accountexpire.sh daily'  #
# to test what happens.                                        #
#                                                              #
# You can set both enable and disable times on a user. For     #
# example, disable date in 5 days and enable date in 10 days.  #
# You can NOT set different flags for a: and r: on the same    #
# user.                                                        #
#                                                              #
# The script can be executed from shell as well as from glftpd.#
#                                                              #
# Run it without arguments for help.                           #
#                                                              #
# Never run it with argument 'daily' unless you mean it.       #
#                                                              #
# Users are never removed automatically from the DB unless     #
# they are purged from the site.                               #
#                                                              #
#-[ Changelog ]------------------------------------------------#
#                                                              #
# 1.1   : Add: Default flag can now be specified with f:<flag> #
#              when using add.                                 #
#                                                              #
#         Add: Added option PERSISTENT. Previously it would    #
#              only do the things you've set if it was the     #
#              exact day you set it to do it. With PERSISTENT  #
#              on TRUE, it will check that its done each day   #
#              afterwards too ( see info about it above ).     #
#                                                              #
#         Cng: The e: and s: options were apparently confusing #
#              so its now a: for when to (a)dd the the flag &  #
#              r: for when to (r)emove the flag.               #
#              Helptext was redone to better reflect usage.    #
#                                                              #
#         Fix: You could execute 'daily' from glftpd. Not too  #
#              good.                                           #
#                                                              #
# 1.0   : First non beta release.                              #
#                                                              #
#-[ Settings ]-------------------------------------------------#

GLROOT=/glftpd
USERSDIR=/ftp-data/users
SYSLOG=/ftp-data/logs/sysop.log
TMP=/tmp
BYEFILES=/ftp-data/byefiles
DB=/etc/account.expire.db
DATEBIN=
USEFLAG=6
LINEUP=TRUE
PERSISTENT=FALSE


#-[ Script Start ]---------------------------------------------#

## Are we in glftpd or shell? If shell, add GLROOT to all paths.
if [ -z "$FLAGS" ]; then
  MODE="shell"
  USERSDIR=$GLROOT$USERSDIR
  TMP=$GLROOT$TMP
  if [ "$SYSLOG" ]; then
    SYSLOG=$GLROOT$SYSLOG
  fi
  if [ "$BYEFILES" ]; then
    BYEFILES=$GLROOT$BYEFILES
  fi
  if [ "$DATEBIN" ]; then
    DATEBIN=$GLROOT$DATEBIN
  fi
  DB=$GLROOT$DB
else
  MODE="glftpd"
fi

## If DATEBIN isnt set, use date from path.
if [ -z "$DATEBIN" ]; then
  DATEBIN="date"
fi

## Verify existance and write perms on DB
if [ ! -e "$DB" ]; then
  echo "Datadase: $DB - Not found. Create it and set chmod 777 on it."
  exit 1
elif [ ! -r "$DB" ]; then
  echo "Database: $DB - No write permissions. Set chmod 777 on it."
  exit 1
fi

## TMP exists, right ?
if [ ! -d "$TMP" ]; then
  echo "Error. TMP dir $TMP does not exist or is not a directory."
  exit 1
fi

## USERSDIR exists, right ?
if [ ! -d "$USERSDIR" ]; then
  echo "Error. USERSDIR dir $USERSDIR does not exist or is not a directory."
  exit 1
fi

## If BYEFILES is set, it exists, right ?
if [ "$BYEFILES" ]; then
  if [ ! -d "$BYEFILES" ]; then
    echo "Error. BYEFILES dir $BYEFILES does not exist or is not a directory."
    exit 1
  fi
fi

## If SYSLOG is set, can we write to it ?
if [ "$SYSLOG" ]; then
  if [ ! -w "$SYSLOG" ]; then
    echo "Error. Cant find, read or write (either) to SYSLOG: $SYSLOG. Check path and perms."
    exit 1
  fi
fi

## Procedure for showing help.
proc_help() {
  echo "#-------------------------------------------------"
  echo "# Account Expire System $VER by Turranius - 2003"
  echo "#-------------------------------------------------"
  echo "# date is in YYYY-MM-DD format"
  echo "# Default flag (if no f: argument) is: $USEFLAG"
  echo "#"
  echo "# Usage:"
  echo "# add <user> r:<startdate / # of days> | a:<enddate / # of days> f:<flag>"
  echo "#            (r = remove flag then. a = add flag then)."
  echo "# del <user>"
  echo "# status (user)"
  echo "# To change expiration or start date, add the user again."
  echo "#"
  echo "# Example: add turran a:2003-10-08 f:4"
  echo "#          Gives turran flag 4 on 2003-10-08"
  echo "# Example: add turran a:2003-10-10 r:2003-10-20"
  echo "#          Gives turran flag $USEFLAG on 2003-10-10 and remove it on 2003-10-20"
}


## Procedure for centering text left, 14 chars top (the # of dots)
proc_cleft() {
  SPACE=' '
  while [ -z "$( echo "$VARIABLE" | grep .............. )" ]; do
    VARIABLE="$VARIABLE$SPACE"
  done
}

## Procedure for grabbing data from DB
proc_grabdata() {
  USERDATATMP=`echo "$USERDATA" | tr -s '^' ' '`
  for each in $USERDATATMP; do
    if [ "$( echo "$each" | grep "^start:" )" ]; then
      DBSTART=`echo "$each" | cut -d ':' -f2`
    fi
    if [ "$( echo "$each" | grep "^end:" )" ]; then
      DBEND=`echo "$each" | cut -d ':' -f2`
    fi
    if [ "$( echo "$each" | grep "^flag:" )" ]; then
      USEFLAG=`echo "$each" | cut -d ':' -f2`
      FLAGOVERRIDE="TRUE"
    fi
  done

  
  echo "Info: $CURUSER's info in DB $WASIS:"
  if [ "$DBSTART" ]; then
    echo "Give (+) date $WASIS set to $DBSTART"
  fi
  if [ "$DBEND" ]; then
    echo "Remove (-) date $WASIS set to $DBEND"
  fi
  if [ "$FLAGOVERRIDE" = "TRUE" ]; then
    echo "Flag $WASIS set to: $USEFLAG"
  fi
}

## If a hard date is entered, verify that its correct !
proc_verifydate() {
  if [ -z "$( echo "$CHECKDATE" | grep "^20[0-1][2-9]\-[0-1][0-9]\-[0-3][0-9]$" )" ]; then
    proc_help
    echo " "
    echo "Error. Not acceptable date: $CHECKDATE"
    exit 1
  fi

  MONTHEXPIRE=`echo "$CHECKDATE" | cut -d '-' -f2`
  DAYEXPIRE=`echo "$CHECKDATE" | cut -d '-' -f3`
  if [ "$MONTHEXPIRE" = "00" -o "$DAYEXPIRE" = "00" ]; then
    proc_help
    echo " "
    echo "Error. month or day cant be 00."
    exit 1
  fi

  case $MONTHEXPIRE in
    01) MONTHTOTAL="31" ;;
    02) MONTHTOTAL="28" ;;
    03) MONTHTOTAL="31" ;;
    04) MONTHTOTAL="30" ;;
    05) MONTHTOTAL="31" ;;
    06) MONTHTOTAL="30" ;;
    07) MONTHTOTAL="31" ;;
    08) MONTHTOTAL="31" ;;
    09) MONTHTOTAL="30" ;;
    10) MONTHTOTAL="31" ;;
    11) MONTHTOTAL="30" ;;
    12) MONTHTOTAL="31" ;;
    *) echo "Error. Only 12 months in a year!"; exit 1 ;;
  esac
  
  if [ "$DAYEXPIRE" -gt "$MONTHTOTAL" ]; then
    proc_help
    echo " "
    echo "Error. Only $MONTHTOTAL days in month $MONTHEXPIRE"
    exit 1
  fi

  MONTHSECNOW=`$DATEBIN +%s`
  MONTHSECADD=`$DATEBIN -d "$CHECKDATE" +%s`
  if [ "$MONTHSECNOW" -gt "$MONTHSECADD" ]; then
    echo "Error. $CHECKDATE is in the past or today. Only add future dates!"
    echo "If you want to force 'today' anyway, just use a 0 as date."
    exit 1
  fi
  unset MONTHSECNOW; unset MONTHSECADD
}

## Procedure for writing to syslog, if set.
proc_syslog() {
  if [ "$SYSLOG" ]; then
    echo "$($DATEBIN +%a" "%b" "%e" "%T" "%Y)" "[A-Expire] $@" >> $SYSLOG
  fi
}

## Procedure for adding flag to user at the expire period.
proc_addflag() {
  CURFLAGS="$( grep "^FLAGS " $USERSDIR/$CURUSER | cut -d ' ' -f2 )"

  if [ "$CURFLAGS" ]; then
    if [ -z "$( echo $CURFLAGS | grep "$USEFLAG" )" ]; then
      sed -e "s/^FLAGS $CURFLAGS/FLAGS $USEFLAG$CURFLAGS/" $USERSDIR/$CURUSER > $TMP/$CURUSER.TMP
      cp -f $TMP/$CURUSER.TMP $USERSDIR/$CURUSER
      rm -f $TMP/$CURUSER.TMP

      proc_syslog "'$CURUSER' +$USEFLAG flag (added)."

      if [ "$BYEFILES" ]; then
        echo "Your account expired on $END" >> $BYEFILES/$CURUSER.bye
      fi

    fi
  fi
}

## Procedure for removing flag from user when start period is entered.
proc_removeflag() {
  CURFLAGS="$( grep "^FLAGS " $USERSDIR/$CURUSER | cut -d ' ' -f2 )"

  if [ "$CURFLAGS" ]; then
    if [ "$( echo "$CURFLAGS" | grep "$USEFLAG" )" ]; then
      NEWFLAGS=`echo "$CURFLAGS" | tr -d "$USEFLAG"`
      sed -e "s/^FLAGS $CURFLAGS/FLAGS $NEWFLAGS/" $USERSDIR/$CURUSER > $TMP/$CURUSER.TMP
      cp -f $TMP/$CURUSER.TMP $USERSDIR/$CURUSER
      rm -f $TMP/$CURUSER.TMP

      proc_syslog "'$CURUSER' -$USEFLAG flag (removed)."

    fi
  fi
}

## Procedure for adding users to DB.
proc_addexpire() {
  if [ -z "$A1" -o -z "ADATA" ]; then
    proc_help
    exit 1
  fi

  CURUSER="$A2"
  for each in $DATA; do
    if [ "$( echo "$each" | grep "^r:" )" ]; then
      START=`echo "$each" | cut -d ':' -f2`
    fi
    if [ "$( echo "$each" | grep "^a:" )" ]; then
      END=`echo "$each" | cut -d ':' -f2`
    fi
    if [ "$( echo "$each" | grep "^f:" )" ]; then
      USEFLAG=`echo "$each" | cut -d ':' -f2`
      if [ -z "$USEFLAG" ]; then
        echo "f: detected, but no flag specified..."
        exit 1
      fi
    fi
  done

  ## Make sure flag given is in uppercase.
  USERFLAG=`echo $USEFLAG | tr '[:lower:]' '[:upper:]'`

  if [ -z "$START" -a -z "$END" ]; then
    proc_help
    echo " "
    echo "Error. Got neither start nor end time. Aborting."
    exit 1
  fi

  if [ ! -e "$USERSDIR/$CURUSER" ]; then
    echo "Error. $CURUSER does not exist. Add the user first."
    exit 1
  fi

  if [ "$START" ]; then
    if [ "$( echo "$START" | grep "\-.*\-" )" ]; then
      CHECKDATE="$START"
      proc_verifydate

      echo "Selected: $CURUSER gets flag $USEFLAG removed on $START"

    else

      if [ "$( echo "$START" | tr -d '[:digit:]' )" ]; then
        proc_help
        echo " "
        echo "Error. When selecting number of days, only use numbers."
        exit 1
      fi

      echo "Selected: $CURUSER gets flag $USEFLAG in for $START days."
      START=`$DATEBIN -d "+$START day" +%Y"-"%m"-"%d`
      echo "Setting enable date to: $START"

    fi
  fi

  if [ "$END" ]; then
    if [ "$( echo "$END" | grep "\-.*\-" )" ]; then
      CHECKDATE="$END"
      proc_verifydate

      echo "Selected: $CURUSER will get flag $USEFLAG on $END"

    else

      if [ "$( echo "$END" | tr -d '[:digit:]' )" ]; then
        proc_help
        echo " "
        echo "Error. When selecting number of days, only use numbers."
        exit 1
      fi

      echo "Selected: $CURUSER will get flag $USEFLAG in $END days."
      END=`$DATEBIN -d "+$END day" +%Y"-"%m"-"%d`
      echo "Setting lock date to: $END"

    fi
  fi

  USERDATA=`grep "^$CURUSER^" $DB | head -n1`
  if [ "$USERDATA" ]; then
    WASIS="was"
    newflag="$USEFLAG"

    proc_grabdata

    USEFLAG="$newflag"
    unset newflag

    grep -v "^$CURUSER^" $DB > $TMP/account.expire.tmp
    cp -f $TMP/account.expire.tmp $DB
    rm -f $TMP/account.expire.tmp
  fi

  echo "Set! Writing $CURUSER's info to DB"
  if [ "$START" ]; then
    START="^start:$START"
  fi
  if [ "$END" ]; then
    END="^end:$END"
  fi
  FLAG="^flag:$USEFLAG"
  echo "$CURUSER$START$END$FLAG" >> $DB
}

## Procedure for delling users from expire DB
proc_delexpire() {
  CURUSER="$A2"

  if [ -z "$CURUSER" ]; then
    proc_help
    exit 0
  fi

  USERDATA=`grep "^$CURUSER^" $DB | head -n1`
  if [ "$USERDATA" ]; then
    WASIS="was"
    proc_grabdata

    grep -v "^$CURUSER^" $DB > $TMP/account.expire.tmp
    cp -f $TMP/account.expire.tmp $DB
    rm -f $TMP/account.expire.tmp
    echo "All start and end data for $CURUSER removed."
  else
    echo "Eh? $CURUSER's account is not set up to expire."
    exit 1
  fi
}

## Procedure for showing status of all in the expire DB
proc_status() {
  CURUSER="$A2"
  if [ "$CURUSER" ]; then
    USERDATA=`grep "^$CURUSER^" $DB | head -n1`
    if [ "$USERDATA" ]; then
      WASIS="is"
      proc_grabdata
    else
      echo "Eh? $CURUSER's account is not set up to expire."
      exit 1
    fi
  else
    for each in `cat $DB`; do

      unset CURUSER; unset START; unset END; unset crap
      each=`echo "$each" | tr -s '^' ' '`

      for crap in $each; do

        if [ -z "$CURUSER" ]; then
          if [ "$LINEUP" = "TRUE" ]; then
            VARIABLE="$crap"
            proc_cleft
            CURUSER="$VARIABLE"
          else
            CURUSER="$crap"
          fi
        fi

        if [ "$( echo "$crap" | grep "^start:" )" ]; then
          START=`echo "$crap" | cut -d ':' -f2`
        fi
        if [ "$( echo "$crap" | grep "^end:" )" ]; then
          END=`echo "$crap" | cut -d ':' -f2`
        fi
        if [ "$( echo "$crap" | grep "^flag:" )" ]; then
          USEFLAG=`echo "$crap" | cut -d ':' -f2`
        fi
      done
      if [ -z "$START" ]; then
        if [ "$LINEUP" = "TRUE" ]; then
          START="N/A       "
        else
          START="N/A"
        fi
      fi
      if [ -z "$END" ]; then
        if [ "$LINEUP" = "TRUE" ]; then
          END="N/A       "
        else
          END="N/A"
        fi
      fi
      echo "$CURUSER Give Flag Date: $START  Remove Flag Date: $END - Flag Used: $USEFLAG"
    done
    if [ -z "$CURUSER" ]; then
      echo "No accounts set for enable or expire."
    fi
  fi
}

## Procedure for running daily. Heres where it does the bad or good stuff.
proc_daily() {
  if [ "$MODE" = "glftpd" ]; then
    echo "Error. Shell only function."
    exit 1
  fi

  DATENOW=`$DATEBIN +%Y"-"%m"-"%d`

  if [ "$PERSISTENT" = "TRUE" ]; then
    DATENOWSEC=`$DATEBIN +%s`
  fi

  for each in `cat $DB`; do
    unset CURUSER; unset START; unset END; unset crap
    each=`echo "$each" | tr -s '^' ' '`

    for crap in $each; do
      if [ -z "$CURUSER" ]; then
        CURUSER="$crap"
      fi
      if [ "$( echo "$crap" | grep "^start:" )" ]; then
        START=`echo "$crap" | cut -d ':' -f2`
      fi
      if [ "$( echo "$crap" | grep "^end:" )" ]; then
        END=`echo "$crap" | cut -d ':' -f2`
      fi
      if [ "$( echo "$crap" | grep "^flag:" )" ]; then
        USEFLAG=`echo "$crap" | cut -d ':' -f2`
      fi
    done

    if [ ! -e "$USERSDIR/$CURUSER" ]; then
      ## Remove purged user from db.
      grep -v "^$CURUSER^" $DB > $TMP/account.expire.tmp
      cp -f $TMP/account.expire.tmp $DB
      rm -f $TMP/account.expire.tmp
    else

      if [ "$START" ]; then
        if [ "$DATENOW" = "$START" ]; then
          proc_addflag
        else
          if [ "$PERSISTENT" = "TRUE" -a "$END" = "" ]; then
            ## Check if this event was in the past.
            STARTSEC=`$DATEBIN -d "$START" +%s`
            if [ "$STARTSEC" -lt "$DATENOWSEC" ]; then
              proc_addflag
            fi
          fi
        fi
      fi

      if [ "$END" ]; then
        if [ "$DATENOW" = "$END" ]; then
          proc_removeflag
        else
          if [ "$PERSISTENT" = "TRUE" -a "$START" = "" ]; then
            ## Check if this event was in the past.
            STARTSEC=`$DATEBIN -d "$START" +%s`
            if [ "$STARTSEC" -lt "$DATENOWSEC" ]; then
              proc_removeflag
            fi
          fi
        fi
      fi

    fi
  done
}

## Set some variables.
A1="$1"
A2="$2"
DATA=`echo "$@" | cut -d ' ' -f2-`

## Main menu, so to speak.
case $A1 in
  add) proc_addexpire; exit 0;;
  del) proc_delexpire; exit 0;;
  daily) proc_daily; exit 0;;
  status) proc_status; exit 0;;
  *) proc_help; exit 0
esac
