#!/bin/bash
VER=1.5

#--[ Info ]--------------------------------------------------------------#
#                                                                        #
# A script to verify the integrity between users/passwd/group files.     #
# It will start with verifying that all entries in the passwd has an     #
# userfile.                                                              #
# It will then check that all users has an entry in the passwd file.     #
# After that, it will check that the primary group in the userfile is    #
# the one specified in the passwd file that user.                        #
# Finally, it will go through the passwd and verify that the UIDs are    #
# listed in order and are unique.                                        #
#                                                                        #
# It will NOT touch your userfiles or passwd file for safety.            #
# It will create a new passwd file which you can overwrite with your old #
# one, once you verified it.                                             #
#                                                                        #
# Just change the settings below and chmod this script to 700 or so. Run #
# it as root.                                                            #
#                                                                        #
# 1.5  : Added another check. Each used group in the userfiles will be   #
#        checked against the groups file to make sure each one exists in #
#        it.                                                             #
#                                                                        #
# 1.4.2: It will not resort the file and say it did unless it finds an   #
#        error. Thanks _n__m                                             #
#                                                                        #
# 1.4.1: The checking if a user had the same uid as another user could   #
#        get messed up if there was a '.' in the users hashed password.  #
#        Fixed, but it now takes a lot longer to run the check. Not like #
#        this is something thats executed daily, so.. fine with me.      #
#                                                                        #
# 1.4  : Now checks the UID's of the users are sorted in the passwd.     #
#        If a user has a higher uid then someone below them in the passwd#
#        then nobody below that user can be deleted / purged.            #
#        Previously it automatically resorted the file on any other      #
#        errors but now checks this too.                                 #
#                                                                        #
#        It will also check that no two users has the same uid.          #
#                                                                        #
# 1.3  : Ok.. added GID_ON_NOGROUP setting. This will control what it    #
#        sets in the passwd for users with no group in the userfiles.    #
#        Most people can leave it at 100 which is NoGroup. However, if   #
#        want another GID, or leaving it empty, you can set that here.   #
#                                                                        #
#        Setting a number will set that number in the passwd.            #
#        Setting it empty will leave the GID field empty in the passwd.  #
#        Setting it to "SKIP" will simply skip users with no group.      #
#                                                                        #
# 1.2  : Added passwd/users verification first. Inconsistencies here     #
#        could lead to an empty passwd file.                             #
#                                                                        #
#        No longer overwrites the current passwd file. Instead, it makes #
#        a new one. Specify where with NEW_PASSWD=                       #
#                                                                        #
#        'fix' argument is no longer needed. A new file will be created  #
#        if needed.                                                      #
#                                                                        #
#        If a user has no group in the userfile, it will check that its  #
#        gid 100 in the passwd file. Previously, the user was skipped.   #
#                                                                        #
#        Thanks to CyberScream for the info based on users in no groups. #
#        Thanks to Maxpecks for getting an empty passwd file and testing #
#        1.2 with me. =)                                                 #
#                                                                        #
# 1.1  : passwd file is now resorted after being fixed.                  #
#        Run argument 'resort' to force a resort (of a previously        #
#        fixed passwd file).                                             #
#        This is a must or the user can not ever be purged incase he is  #
#        'fixed' with this script. Thanks CyberScream for the info.      #
#                                                                        #
#-[ Settings ]-----------------------------------------------------------#

## Path to users dir. Wont be touched
USERS=/glftpd/ftp-data/users

## Path to passwd file. Wont be touched
PASSWD=/glftpd/etc/passwd

## Where to create the new passwd file. Will be touched =)
NEW_PASSWD=/glftpd/etc/passwd.new

## What GID to set to users in no group.
## <a number> = gid to set. "" = Set empty (::). "SKIP" = Just skip them.
## Leave it at 100. "" is NOT recomended.
GID_ON_NOGROUP="100"


#--[ Script Start ]---------------------------------------#

if [ ! -d "$USERS" ]; then
  echo "Error. USERS dir does not exist or isnt a dir: $USERS"
  exit 0
fi

if [ ! -e "$PASSWD" ]; then
  echo "Error. passwd does not exist in defined path: $PASSWD"
  exit 0
else
  groupfile=`dirname $PASSWD`
  if [ ! -e "$groupfile/group" ]; then
    echo "Error. group file not found. Should be in same dir as the passwd, ie $groupfile"
    exit 0
  fi

  cp -f "$PASSWD" "$NEW_PASSWD"
  if [ ! -e "$NEW_PASSWD" ]; then
    echo "Error. Couldnt copy $PASSWD to $NEW_PASSWD for some reason. Quitting."
    exit 1
  fi
fi

echo "--[ Checking that all users in passwd exists as userfiles ]--"
for line in `cat $NEW_PASSWD | cut -d ':' -f1`; do
  if [ ! -e $USERS/$line ]; then
    echo "$line has a passwd entry, but no userfile."
    foundone="YES"
    errors="TRUE"
  fi
done
if [ "$foundone" != "YES" ]; then
  echo "All users in passwd file has a userfile"
fi

echo "--[ Checking that all user files exist in the passwd ]--"

unset foundone
cd $USERS

for user in `grep "^FLAGS " * | cut -d ':' -f1 | grep -v "default.user" `; do
  if [ -z "`cat "$NEW_PASSWD" | grep "^$user\:"`" ]; then
    echo "$user has a userfile, but no passwd entry!"
    foundone="YES"
    errors="TRUE"
  fi
done
if [ "$foundone" != "YES" ]; then
  echo "All userfiles match a user in passwd file"
  echo ""
fi

if [ "$errors" = "TRUE" ]; then
  echo "I will not check the passwd GID's until the above errors are sorted. Quitting now."
  exit 0
fi

proc_resort() {
  cat $NEW_PASSWD | tr ':' ' '| sort -k3,3 -n | tr ' ' ':' > /tmp/passwd.tmp
  if [ ! -e "/tmp/passwd.tmp" ]; then
    echo "Error. Can not find the newly resorted passwd file in /tmp/passwd.tmp"
    exit 1
  else
    if [ -z "`grep "." /tmp/passwd.tmp`" ]; then
      echo "Error. Cant read anything from /tmp/passwd.tmp. Did it get created ok?"
      exit 1
    fi
  fi

  cp -f /tmp/passwd.tmp $NEW_PASSWD
  rm -f /tmp/passwd.tmp

  echo "passwd file resorted ok."
}

if [ "$1" = "resort" ]; then
  proc_resort
  exit 0
fi

proc_fixnogroup() {
  if [ "$GID_ON_NOGROUP" = "SKIP" ]; then
    echo "$user has no group. Skipping that user."
  else

    if [ "`grep "^$user:" $NEW_PASSWD | cut -d ':' -f4`" != "$GID_ON_NOGROUP" ]; then
      echo "$user has no primary group in groupfile. GID in passwd isnt :$GID_ON_NOGROUP: - Fixing."

      proc_getvalues

      echo "Old: $rawdata"
      echo "New: $user:$pass:$uid:$GID_ON_NOGROUP:$dateadded:$homedir:$crap"

      grep -v "$rawdata" $NEW_PASSWD > /tmp/passwd.tmp
      cp -f /tmp/passwd.tmp $NEW_PASSWD
      echo "$user:$pass:$uid:$GID_ON_NOGROUP:$dateadded:$homedir:$crap" >> $NEW_PASSWD
    fi
  fi
}

proc_getvalues() {
  rawdata=`grep "^$user:" $NEW_PASSWD`
  pass=`echo "$rawdata" | cut -d ':' -f2`
  uid=`echo "$rawdata" | cut -d ':' -f3`
  pregid=`echo "$rawdata" | cut -d ':' -f4`
  dateadded=`echo "$rawdata" | cut -d ':' -f5`
  homedir=`echo "$rawdata" | cut -d ':' -f6`
  crap=`echo "$rawdata" | cut -d ':' -f7`
}

echo "--[ Starting GID check on users/passwd/group files ]--"

for user in `grep "^FLAGS " * | cut -d ':' -f1 | grep -v "default.user" `; do
  groupnow=`grep "^GROUP " $user | head -n1 | cut -d ' ' -f2`
  if [ "$groupnow" ]; then
    gidinpasswd=`grep "^$user:" $NEW_PASSWD | cut -d ':' -f4`
    gidingroup=`grep "^$groupnow:" $groupfile/group | cut -d ':' -f3`

    if [ "$gidinpasswd" != "$gidingroup" ]; then
      echo ""
      echo "$user: $groupnow has >$gidingroup< in group file but $user has >$gidinpasswd< in passwd"

      proc_getvalues
      echo "Old: $rawdata"
      echo "New: $user:$pass:$uid:$gidingroup:$dateadded:$homedir:$crap"

      grep -v "$rawdata" $NEW_PASSWD > /tmp/passwd.tmp
      cp -f /tmp/passwd.tmp $NEW_PASSWD
      echo "$user:$pass:$uid:$gidingroup:$dateadded:$homedir:$crap" >> $NEW_PASSWD
    fi
  else
    proc_fixnogroup
  fi
done

echo "--[ Checking that UIDs are listed in order - This takes a while. ]--"
lastuseruid="-1"
for userline in `cat $PASSWD`; do
  username="`echo "$userline" | cut -d ':' -f1`"
  useruid="`echo "$userline" | cut -d ':' -f3`"
  if [ "$useruid" -lt "$lastuseruid" ]; then
    echo "Error. UID for $username ($useruid) is lower then $lastusername ($lastuseruid). Your passwd needs resorting."
    need_resort=true
  fi
  for rawdata in `cat $PASSWD | grep -v "^$username\:"`; do
    temp_uid="`echo "$rawdata" | cut -d ':' -f3`"
    if [ "$temp_uid" = "$useruid" ]; then
      temp_user="`echo "$rawdata" | cut -d ':' -f1`"

      echo "Error. UID of $username ($useruid) matches $temp_user ($temp_uid)"
      echo "This will have to be fixed manually."
      echo ""
      MANUAL_ERROR="TRUE"
    fi
  done
  lastusername="$username"
  lastuseruid="$useruid"

done

echo "--[ Checking that every group used exists in $groupfile/group ]--"
if [ -e "/tmp/verifygroupgid.tmp" ]; then
  rm -f "/tmp/verifygroupgid.tmp"
fi
for groupname in `grep "^GROUP " * | cut -d ' ' -f2`; do 
  echo "$groupname" >> /tmp/verifygroupgid.tmp
done
for groupname in `cat "/tmp/verifygroupgid.tmp" | sort -u`; do
  if [ -z "`grep "^${groupname}:" $groupfile/group`" ]; then
    echo "Warning: Group \"$groupname\" has no entry in $groupfile/group"
    echo "This will need to be fixed manually."
    echo "Users in this group are:"
    for user in `grep "^GROUP $groupname" * | tr ' ' '^'`; do
      username="`echo "$user" | cut -d ':' -f1`"
      group="`echo "$user" | cut -d '^' -f2`"
      if [ "$group" = "$groupname" ]; then
        echo "$username"
      fi
    done
    echo ""
    MANUAL_ERROR=TRUE
  fi
done
rm -f "/tmp/verifygroupgid.tmp"
 
if [ "$need_resort" != "true" ]; then
  if [ "$MANUAL_ERROR" = "TRUE" ]; then
    echo "There were errors that needs manual attention."
  else
    echo "All users are ok!"
  fi
  rm -f "$NEW_PASSWD"
  exit 0
else
  proc_resort
  echo "A new passwd created in $NEW_PASSWD"
  if [ "$MANUAL_ERROR" = "TRUE" ]; then
    echo "There were errors that needs manual attention."
  fi
  exit 0
fi
