
#Region ;**** Directives created by AutoIt3Wrapper_GUI ****
#AutoIt3Wrapper_Outfile=Tur-Devcon_Game_Controller_Order.exe
#AutoIt3Wrapper_Res_Fileversion=1.2.0.0
#AutoIt3Wrapper_Res_ProductVersion=1.2.0
#EndRegion ;**** Directives created by AutoIt3Wrapper_GUI ****

#RequireAdmin
#include <Constants.au3>
#include <Array.au3>

Local $Version = "1.2"

; Find out which devcon to run depending on OS architecture.
If @OSArch = "X64" Then
	Global $DevCon = @ScriptDir & "\devcon64.exe"
ElseIf @OSArch = "X86" Then
	Global $DevCon = @ScriptDir & "\devcon32.exe"
Else
	MsgBox(16, "Error", "Could not determine if this is an x64 or x86 machine.. ?!")
	Exit
EndIf

; Define the INI file.
Global $IniFile = @ScriptDir & "\Settings.ini"

; Make sure we really have a devcon command...
If Not FileExists($DevCon) Then
	MsgBox(16, "Error", "Could not find " & $DevCon)
	Exit
EndIf

; Check if the ini file exists. If not, run the guide.
If Not FileExists($IniFile) Then

	; Find all HIDs for everything called "game controller".
	Local $CurrentPID = Run(@ComSpec & ' /c ' & Chr(34) & $DevCon & Chr(34) & " status *game controller* | findstr HID\VID", "", @SW_HIDE, $STDERR_CHILD + $STDOUT_CHILD)
	If Not ProcessWaitClose($CurrentPID, 60) Then
		MsgBox(16, "No output", "Did not get any return from " & $DevCon & " status *game controller*")
		Exit
	EndIf

	; Read the output we got.
	$CurrentOutput = StdoutRead($CurrentPID)
	If $CurrentOutput == "" Then
		MsgBox(16, "Error", "Did not find any game controllers when running" & @CRLF & Chr(34) & $DevCon & Chr(34) & " status *game controller*" & @CRLF & @CRLF & "Make sure your BlissBox is connected.")
		Exit
	EndIf

	; Run the Game Controllers control panel item.
	Run("control.exe joy.cpl")

	; Show initial message with information.
	MsgBox(0, "Setup", "This is the first time you run this program. We will now run through the setup and identify your joystick ports HIDs." & @CRLF & @CRLF & "I have opened the Game Controllers window and you should see 4 controllers (4-Play_port1 to port4)." & @CRLF & @CRLF & "To make this easier, if you see any other game controllers, disconnect them for now before clicking OK.")

	; Create a new array from the output from devcon, clean up of any newlines and empty rows that might appear.
	Local $HIDArrayRaw[0]
	Local $HIDArrayRawRaw = StringSplit($CurrentOutput, @CR)
	For $i = 1 To $HIDArrayRawRaw[0]
		Local $HIDCleaned = StringStripWS($HIDArrayRawRaw[$i], 1)
		If $HIDCleaned <> "" Then
			_ArrayAdd($HIDArrayRaw, $HIDCleaned)
		EndIf
	Next
	Local $HIDsFound = UBound($HIDArrayRaw)

	; The final list of HIDs should be 4 (or more). Make sure it is.
	If $HIDsFound < "4" Then
		MsgBox(16, "Failure", "Did not find atleast 4 HIDs. Aborting.")
		Exit
	EndIf

	; Craete a nice list we can use in the next msgbox.
	Local $HIDDisplay = ""
	For $i = 0 To $HIDsFound - 1
		If $HIDDisplay = "" Then
			$HIDDisplay = $HIDArrayRaw[$i]
		Else
			$HIDDisplay = $HIDDisplay & @CRLF & $HIDArrayRaw[$i]
		EndIf
	Next

	; Show the next message, that we are ready to start.
	Local $GoResponse = MsgBox(49, "Ready!", "I am now going to disconnect the game controllers, one at a time." & @CRLF & @CRLF & "Have a look at the Game Controllers window and tell me which port disappears when you click OK." & @CRLF & @CRLF & "FYI: The following HIDs were found:" & @CRLF & $HIDDisplay)
	; Make sure OK was clicked. Quit otherwise.
	If $GoResponse <> "1" Then
		Exit
	EndIf

	; Go through each HID in the Settings.ini file.
	For $i = 0 To $HIDsFound - 1
		; Put the HID in a string (easier to work with).
		Local $HIDCleaned = $HIDArrayRaw[$i]
		; Make sure its not empty. It should never be, but here is a check for it since we can.
		If $HIDCleaned <> "" Then

			; Make sure the HID contains atleast 2 backslashes or its nothing we want.. weird
			If Not StringInStr($HIDCleaned, "\", 0, 2) Then
				MsgBox(16, "HID error", "The HID:" & @CRLF & $HIDCleaned & @CRLF & "does not contain atleast 2 backslashes. Aborting")
				_Exit()
			EndIf

			; Create a temporary array from the HID. We cant use the full HID as we only want the middle part in the backslashes.
			Local $HIDArrayTemp = StringSplit($HIDCleaned, "\")
			If Not IsArray($HIDArrayTemp) Then
				MsgBox(16, "Error", "Could not split up " & $HIDCleaned & " - Aborting")
				_Exit()
			EndIf
			; Change the $HIDCleaned string to only be the middle part of the backslashes.
			Local $HIDCleaned = $HIDArrayTemp[2]

			; Run devcon and remove this HID.
			RunWait(Chr(34) & $DevCon & Chr(34) & " remove " & Chr(34) & "*" & $HIDCleaned & "*" & Chr(34), "", @SW_HIDE)
			; Ask which port vanished from the list. 1-4
			Local $SelectedPort = InputBox("Select a port", "Which port vanished from the Game Controllers list? Specify 1 - 4 or 0 if none of them did.")

			; If we got no reponse from the user, abort using _Exit() which rescans the hardware first.
			If $SelectedPort == "" Then
				MsgBox(16, "Aborting.", "Nothing typed. Running rescan and aborting")
				If FileExists($IniFile) Then
					FileDelete($IniFile)
				EndIf
				_Exit()
			EndIf

			; Rescan the hardware here so it pops back in the list.
			_RescanHardware()

			; Depending on what the user responded with, write it to the Settings.ini file if it was valid or show a message if it was not.
			If $SelectedPort == "0" Then
				MsgBox(0, "Hmm", "No controllers vanished? I will continue but its quite possible that you are not running this program as administrator. If you have more then the 4 ports visable in the Game Controllers window and one of the others vanished, this is normal.")
			ElseIf $SelectedPort == "1" Then
				IniWrite($IniFile, "Game Ports", "Port1", $HIDCleaned)
				MsgBox(0, "Port1", "Wrote Port1=" & $HIDCleaned & " to the Settings.ini file." & @CRLF & @CRLF & "I will now remove another port. Look at the Game Controllers window again.")
			ElseIf $SelectedPort == "2" Then
				IniWrite($IniFile, "Game Ports", "Port2", $HIDCleaned)
				MsgBox(0, "Port2", "Wrote Port2=" & $HIDCleaned & " to the Settings.ini file." & @CRLF & @CRLF & "I will now remove another port. Look at the Game Controllers window again.")
			ElseIf $SelectedPort == "3" Then
				IniWrite($IniFile, "Game Ports", "Port3", $HIDCleaned)
				MsgBox(0, "Port3", "Wrote Port3=" & $HIDCleaned & " to the Settings.ini file." & @CRLF & @CRLF & "I will now remove another port. Look at the Game Controllers window again.")
			ElseIf $SelectedPort == "4" Then
				IniWrite($IniFile, "Game Ports", "Port4", $HIDCleaned)
				MsgBox(0, "Port4", "Wrote Port4=" & $HIDCleaned & " to the Settings.ini file." & @CRLF & @CRLF & "I will now remove another port. Look at the Game Controllers window again.")
			Else
				; Something other the 0-4 was typed. Delete ini file and exit. No need to rescan hardware here as its done above anyway.
				MsgBox(16, "Error", "You should only type 0, 1, 2, 3 or 4. Aborting.")
				If FileExists($IniFile) Then
					FileDelete($IniFile)
				EndIf
				Exit
			EndIf
		EndIf
	Next

	; We are done. Do a final rescan of hardware so the last one pops back in again.
	_RescanHardware()
	; Sort the INI file. Looks better.
	_IniSort($IniFile)
	; Show a final message and exit.
	MsgBox(0, "Done", "Setup is done. Providing all went well, you should now be able to restart me to have the ports reordered automatically." & @CRLF & @CRLF & "Please verify that you have 4 ports defined in Settings.ini." & @CRLF & @CRLF & "To restart this setup, delete the Settings.ini file")
	Exit

EndIf

; We end up here if you start the program with a Settings.ini file existing already.

; Read the Ports from the ini file into an array.
Global $PortArray = ""
Global $PortArray = IniReadSection($IniFile, "Game Ports")

; Verify that we got an array and that it is indeed 4 rows.
If Not IsArray($PortArray) Then
	MsgBox(16, "Error", "Settings.ini error. Can not read ports. Please delete Settings.ini and rerun to set it up correctly.")
	Exit
ElseIf $PortArray[0][0] <> "4" Then
	MsgBox(16, "Error", "Error in Settings.ini. Seems there are not 4 ports defined. Please delete Settings.ini and rerun to set it up correctly.")
	Exit
EndIf

; Cleared to go. Remove all except Port 1
For $i = 1 To $PortArray[0][0]
	Local $Name = $PortArray[$i][0]
	Local $ID = $PortArray[$i][1]
	If $Name <> "Port1" And $ID <> "" Then
		ConsoleWrite("Removing " & $Name & " which has ID: " & $ID & @CRLF)
		RunWait(Chr(34) & $DevCon & Chr(34) & " remove " & Chr(34) & "*" & $ID & "*" & Chr(34), "", @SW_HIDE)
	EndIf
Next

; Rescan it
RunWait(Chr(34) & $DevCon & Chr(34) & " rescan", "", @SW_HIDE)

; Remove all except Port 1 and 2
For $i = 1 To $PortArray[0][0]
	Local $Name = $PortArray[$i][0]
	Local $ID = $PortArray[$i][1]
	If $Name <> "Port1" And $Name <> "Port2" And $ID <> "" Then
		ConsoleWrite("Removing " & $Name & " which has ID: " & $ID & @CRLF)
		RunWait(Chr(34) & $DevCon & Chr(34) & " remove " & Chr(34) & "*" & $ID & "*" & Chr(34), "", @SW_HIDE)
	EndIf
Next

; Rescan it
RunWait(Chr(34) & $DevCon & Chr(34) & " rescan", "", @SW_HIDE)

; Remove all except Port 1 and 2 and 3
For $i = 1 To $PortArray[0][0]
	Local $Name = $PortArray[$i][0]
	Local $ID = $PortArray[$i][1]
	If $Name <> "Port1" And $Name <> "Port2" And $Name <> "Port3" And $ID <> "" Then
		ConsoleWrite("Removing " & $Name & " which has ID: " & $ID & @CRLF)
		RunWait(Chr(34) & $DevCon & Chr(34) & " remove " & Chr(34) & "*" & $ID & "*" & Chr(34), "", @SW_HIDE)
	EndIf
Next

; Rescan it
RunWait(Chr(34) & $DevCon & Chr(34) & " rescan", "", @SW_HIDE)

; We are done here. Functions below.




; This is only used when creating the Settings.ini file to show a message when rescanning hardware. Might take a while if a harddrive had spun down, etc..
Func _RescanHardware()
	SplashTextOn("Rescanning hardware", "Please wait...", 200, 50)
	RunWait(Chr(34) & $DevCon & Chr(34) & " rescan", "", @SW_HIDE)
	SplashOff()
EndFunc   ;==>_RescanHardware

Func _Exit()
	_RescanHardware()
	Exit
EndFunc   ;==>_Exit

; Function to sort the ini file at the end of creation. Not needed but looks better in Settings.ini if they are in order from 1-4.
Func _IniSort($hIni)
	Local $aIRSN = IniReadSectionNames($hIni)
	If Not IsArray($aIRSN) Then Return SetError(1, 0, 0)
	_ArraySort($aIRSN, 0, 1)
	Local $aKey, $sHold
	For $iCC = 1 To UBound($aIRSN) - 1
		Local $aIRS = IniReadSection($hIni, $aIRSN[$iCC])
		If Not IsArray($aIRS) Then ContinueLoop
		For $xCC = 1 To $aIRS[0][0]
			$aKey &= $aIRS[$xCC][0] & Chr(1)
		Next
		If $aKey Then
			$aKey = StringSplit(StringTrimRight($aKey, 1), Chr(1))
			_ArraySort($aKey, 0, 1)
			$sHold &= '[' & $aIRSN[$iCC] & ']' & @CRLF
			For $aCC = 1 To UBound($aKey) - 1
				$sHold &= $aKey[$aCC] & '=' & IniRead($hIni, $aIRSN[$iCC], $aKey[$aCC], 'blahblah') & @CRLF
			Next
			$aKey = ''
		EndIf
	Next
	If $sHold Then
		$sHold = StringTrimRight($sHold, 2)
		FileClose(FileOpen($hIni, 2))
		FileWrite($hIni, $sHold)
		Return 1
	EndIf
	Return SetError(1, 0, 0)
EndFunc   ;==>_IniSort
